"""Data Related Utilities."""
import numpy as np
import urllib.request
from pathlib import Path


class OzoNetDataLoader:
    """
    A Simple Data Loader for Keras to load data.
    """

    def __init__(self, X_file_name, y_file_name):
        '''
        Args:
            X_path (String): Path to the Input Images as numpy arrays.
            y_path (String): Path to the Output Images as numpy arrays.
        '''

        # If the file exists in /tmp, we read it.
        # .. if not, we download it to /tmp and read it anyway.
        local_prefix = '/tmp/'
        remote_prefix = 'https://s3.amazonaws.com/phd-interpolation-bucket/data/'

        # if files not in /tmp -> download them to /tmp.
        for filename in [X_file_name, y_file_name]:
            file_path = Path(local_prefix + filename)
            if not file_path.is_file():
                urllib.request.urlretrieve(
                    url=remote_prefix+filename, filename=local_prefix+filename)

        # now we load the data.
        X = np.load(file=local_prefix + X_file_name)
        y = np.load(file=local_prefix + y_file_name)

        # X and y are of the same nature (patches).
        # that's why we will use the same mean and std to normalize them both (and train/val/test sets).
        Xy = np.concatenate((X, y))
        Xy_mean = Xy.mean()
        Xy_std = Xy.std()

        # We will add the channel component to the shape
        X = X.reshape((X.shape[0], 30, 30, 1))
        y = y.reshape((X.shape[0], 30, 30, 1))

        # now we normalize the data.
        self.X = (X - Xy_mean) / Xy_std
        self.y = (y - Xy_mean) / Xy_std

    def __len__(self):
        '''returns the length of the loaded dataset.'''
        return self.X.shape[0]

    def __getitem__(self, idx):
        '''Returns Shallow-SoTa interpolated Image X[i] & its ground truth pollution image y[i].'''
        sample = {'X': self.X[idx], 'y': self.y[idx]}
        return sample
