'''
Daedalus Ionospheric Profile Continuation (DIPCont) Project
DIPContBas.py: DIPCont basic setup and exchange variables.
PYTHON CODE UNDER DEVELOPMENT, PROVIDED AS IS, NO WARRANTY.
Written by Joachim Vogt, Jacobs University Bremen (JUB).
Tested also by Octav Marghitu, ISS Bucharest (ISS), 
Adrian Blagau (ISS, JUB), Leonie Pick (DLR Neustrelitz, JUB), 
and Nele Stachlys (U Potsdam, JUB); in collaboration with 
Stephan Buchert, Swedish Institute of Space Physics in Uppsala, 
Theodoros Sarris, Democritus University of Thrace in Xanthi 
(DUTH), Stelios Tourgaidis (DUTH), Thanasis Balafoutis (DUTH), 
Dimitrios Baloukidis (DUTH), and Panagiotis Pirnaris (DUTH).
This version: v0.2, 2022-07-06.
'''

#.. General imports
import numpy as np
import matplotlib.pyplot as plt
import pandas as pd

###########################################################
### (1) General physical constants and planetary parameters
###########################################################

#.. Physical constants
GravConst = 6.6743e-11  #.. gravitational constant G [m^3/kg/s^2]
IdGasConst = 8.314      #.. ideal gas constant [kg.m^2/s^2/K/mol]
AvoConst = 6.022e23     #.. Avogadro's constant [1/mol]
ElemCharge = 1.6022e-19 #.. elementary charge [C]

#.. Earth's planetary parameters
MEarth = 5.9722e24      #.. Earth's planetary mass M [kg]
REarth = 6371e3         #.. Earth's volumetric radius [m]

#.. Parameters for conductivity modeling
SCin = 5e-19           #.. ion-neutral collision cross-section [m^2]

#####################################################
### (2) LTI region boundaries and regional parameters
#####################################################

#.. LTI region boundaries [m]
zBot = 100e3            #.. lower vertical boundary [m]
zTop = 200e3            #.. upper vertical boundary [m]
xLef = -2000e3          #.. left horizontal boundary [m]
xRig = -xLef            #.. right horizontal boundary [m]

#.. LTI physical variables assumed to be constant and known
MnLTI = 0.028           #.. molar mass of neutrals [kg/mol]
MiLTI = 0.028           #.. molar mass of ions [kg/mol]
BmLTI = 5e-5            #.. magnetic field strength [T]

#.. Default LTI model parameters at the lower boundary
TnBotLef = 200          #.. neutral temperature at (xLef,zBot) [K]
TnBotRig = 200          #.. neutral temperature at (xRig,zBot) [K]
NnBotLef = 1e19         #.. neutral density at (xLef,zBot) [1/m^3]
NnBotRig = 1e19         #.. neutral density at (xRig,zBot) [1/m^3]
TiBotLef = 200          #.. ion temperature at (xLef,zBot) [K]
TiBotRig = 200          #.. ion temperature at (xRig,zBot) [K]

#.. Default LTI model parameters at the upper boundary
TnTopLef = 1000         #.. neutral temperature at (xLef,zTop) [K]
TnTopRig = 1000         #.. neutral temperature at (xRig,zTop) [K]
TiTopLef = 1000         #.. ion temperature at (xLef,zTop) [K]
TiTopRig = 1000         #.. ion temperature at (xRig,zTop) [K]

#.. Default electron density parameters at ionization peak(s)
NeIpk = 1e11            #.. electron density peak value [1/m^3]
NeIpkVal = 1.5*NeIpk    #.. for LTIModelType=='AuroralZoneCrossing'
NeIpkDel = 0.5*NeIpk    #.. horizontal diff in peak density [1/m^3]
zIpk = 110e3            #.. electron density peak altitude [m]
zIpkVal = zIpk          #.. for LTIModelType=='AuroralZoneCrossing'
zIpkDel = 10e3          #.. horizontal difference in peak altitude [m]
HNnIpk = 10e3           #.. density scale height at ionization peak [m]

#.. Further LTI model parameters : horizontal variation of Ne
LTIModelType = 'NeHorizontallyUniform'     #.. no horizontal variation
#---LTIModelType = 'NeAuroralZoneCrossing'     #.. auroral oval crossing

####################################################
### (3) Satellite parameters and instrumental errors
####################################################

#.. Satellite orbit parameters
tBeg = -300             #.. start time [s]
tEnd = -tBeg            #.. end time [s]
ObsPerSec = 16          #.. observations per second
zPerA = 130e3           #.. perigee altitude of lower satellite [m]
zApoA = 3000e3          #.. apogee altitude of lower satellite [m]
zPerB = 150e3           #.. perigee altitude of upper satellite [m]
zApoB = zApoA + zPerA - zPerB  #.. apogee alt. of upper sat. [m]

#.. Daedalus relative errors of single measurements
RelErrTn = 0.2          #.. neutral temperature
RelErrNn = 0.2          #.. neutral density
RelErrNe = 0.1          #.. electron density
RelErrTi = 0.1          #.. ion temperature

#####################################################
### (4) Parameter estimation and profile continuation
#####################################################

### Comparison with nomenclature in the DIPCont paper:
###.... IGHNn, the inverse gradient of neutral density scale
###........ height, corresponds to the parameter eta = eta_n.
###.... IGHNn and the local neutral density scale height HNn0
###........ define the neutral temperature gradient length as
###........ Ln0 = IGHNn*HNn0.
###.... For reasons of computational convenience, the ion
###........ temperature gradient length Li0 is handled in the
###........ same way using analogous parameters IGHNi and HNi0.

#.. Default values of LTI model parameters in local representation
z0 = 150e3              #.. reference altitude [m]
Tn0 = 200               #.. neutral temperature [K]
HNn0 = 10e3             #.. neutral density scale height [m]
IGHNn = 5               #.. inverse density scale height gradient
Ne0 = 1e11              #.. electron density [1/m^3]
Nn0 = 1e17              #.. neutral density [1/m^3]
Lr0cc = 30e3            #.. rad. abs. length * cos(chi) [m]
NeF = 0                 #.. F layer electron density level [1/m^3]
Ti0 = 200               #.. ion temperature [K]
HNi0 = 10e3             #.. ion density scale height [m]
IGHNi = 5               #.. inverse ion density scale height gradient

#.. Default predictions at lower boundary
TnBotPrd = 0.5*(TnBotLef+TnBotRig)
TiBotPrd = 0.5*(TiBotLef+TiBotRig)
TnBotEst = TnBotPrd
TiBotEst = TiBotPrd

#.. Relative errors of boundary estimates
RelErrTnBot = 0.05      #.. neutral temperature at lower boundary
RelErrTiBot = 0.05      #.. ion temperature at lower boundary

#.. Initial estimates
Tn0Init = 500
Nn0Init = 1e11
Ne0Init = 1e5
Lr0ccInit = 30e3
NeFInit = 1e4
Ti0Init = 500

#.. Horizontal coordinates for extrapolation [m]
LenGrd = 9              #.. length of horizontal grid
xGrd = np.linspace(xLef,xRig,LenGrd)    #.. horizontal grid [m]
xWin = 100e3            #.. horizontal selection window [m]

#.. Extrapolation horizons
ExtHorPct = 0.5*2**np.arange(8)
ExtHorCol = 2*['yellow']+2*['orange']+2*['red']+2*['magenta']
ExtHorLst = 4*['dotted','solid']

########################################
### (5) Derived variables and dataframes
########################################

#.. Product of grav constant and planetary mass [m^3/s^2]
GMEarth = GravConst*MEarth
#.. Grav acceleration at the planetary surface [m/s^2]
GravEarth = GMEarth/REarth**2
#.. Ion gyrofrequency (angular frequency) [rad/s]
FGiLTI = ElemCharge*BmLTI*AvoConst/MiLTI

#.. Pandas dataframes for Bot/Top variables at left and right boundaries
dfVarBotLef = pd.Series( [TnBotLef,NnBotLef,TiBotLef],\
                             index=['Tn','Nn','Ti'] )
dfVarBotRig = pd.Series( [TnBotRig,NnBotRig,TiBotRig],\
                             index=['Tn','Nn','Ti'] )
dfVarTopLef = pd.Series( [TnTopLef,TiTopLef],\
                             index=['Tn','Ti'] )
dfVarTopRig = pd.Series( [TnTopRig,TiTopRig],\
                             index=['Tn','Ti'] )

#######################################
### (6) Satellite orbits around perigee
#######################################

def OrbitAtPerPA(zPer,zApo,tBeg=tBeg,tEnd=tEnd):
    '''
    Orbit at perigee : polynomial approximation
    Input:
        zPer : perigee altitude [m]
        zApo : apogee altitude [m]
    Keywords:
        tBeg : start time [s]
        tEnd : stop time [s]
    Output:
        zSat : orbital altitudes [m]
        xSat : horizontal distances [m]
        tSat : time array (at perigee: tSat=0) [s]
    '''
    GM = GMEarth
    RE = REarth
    len_tSat = 1+ObsPerSec*np.int(np.rint(tEnd-tBeg))
    tSat = np.linspace(tBeg,tEnd,len_tSat)
    RPer = RE + zPer
    RApo = RE + zApo
    APer = GM/RPer**2*(RApo-RPer)/(RApo+RPer)
    zSat =  zPer + 0.5*APer*tSat**2
    VPer = np.sqrt(2*GM*RApo/RPer/(RApo+RPer))
    xSat = RE*VPer/RPer*( tSat - APer*tSat**3/(3*RPer) )
    return zSat,xSat,tSat

def OrbitAtPerSV(zPer,zApo,tBeg=tBeg,tEnd=tEnd):
    '''
    Orbit at perigee : Stoermer-Verlet integration
    Input:
        zPer : perigee altitude [m]
        zApo : apogee altitude [m]
    Keywords:
        IntPerSec : number of integration steps per second
        tBeg : start time [s]
        tEnd : stop time [s]
    Output:
        zSat : orbital altitudes [m]
        xSat : horizontal distances [m]
        tSat : time array (at perigee: tSat=0) [s]
    '''
    GM = GMEarth
    RE = REarth
    #.. time array [s]
    len_tSat = 1+ObsPerSec*np.int(np.rint(tEnd-tBeg))
    tSat = np.linspace(tBeg,tEnd,len_tSat)
    #.. index of perigee time t=0
    itSat0 = np.abs(tSat).argmin()
    #.. perigee and apogee geocentric distances [m]
    RPer = RE + zPer
    RApo = RE + zApo
    #.. velocity at perigee
    VPer = np.sqrt(2*GM*RApo/RPer/(RApo+RPer))
    #.. global cartesian coordinates (p,q) and geocentric distance r [m]
    pSat = np.zeros(len_tSat)
    qSat = np.zeros(len_tSat)
    rSat = np.zeros(len_tSat)
    #.. forward integration from perigee
    dtInt = 1/ObsPerSec
    vpOld = 0
    vqOld = VPer
    pOld = RPer
    qOld = 0
    rOld = np.sqrt( pOld**2 + qOld**2 )
    pSat[itSat0] = pOld
    qSat[itSat0] = qOld
    rSat[itSat0] = rOld
    for k in range(itSat0+1,len_tSat):
        vpNew = vpOld - dtInt*GM*pOld/rOld**3
        vqNew = vqOld - dtInt*GM*qOld/rOld**3
        pNew = pOld + dtInt*vpNew
        qNew = qOld + dtInt*vqNew
        vpOld = vpNew
        vqOld = vqNew
        pOld = pNew
        qOld = qNew
        rOld = np.sqrt( pOld**2 + qOld**2 )
        pSat[k] = pOld
        qSat[k] = qOld
        rSat[k] = rOld
    #.. backward integration from perigee
    dtInt = -1/ObsPerSec
    vpOld = 0
    vqOld = VPer
    pOld = RPer
    qOld = 0
    rOld = np.sqrt( pOld**2 + qOld**2 )
    for k in range(itSat0-1,-1,-1):
        vpNew = vpOld - dtInt*GM*pOld/rOld**3
        vqNew = vqOld - dtInt*GM*qOld/rOld**3
        pNew = pOld + dtInt*vpNew
        qNew = qOld + dtInt*vqNew
        vpOld = vpNew
        vqOld = vqNew
        pOld = pNew
        qOld = qNew
        rOld = np.sqrt( pOld**2 + qOld**2 )
        pSat[k] = pOld
        qSat[k] = qOld
        rSat[k] = rOld
    zSat = rSat - RE
    xSat = RE*np.arctan2(qSat,pSat)
    return zSat,xSat,tSat

#.. Compute satellite orbits and store in arrays
zSatA,xSatA,tSatA = OrbitAtPerPA(zPerA,zApoA)
zSatB,xSatB,tSatB = OrbitAtPerPA(zPerB,zApoB)

##############################################################
### (7) Supplementary variables and arrays, e.g., for plotting
##############################################################

#.. Default arrays, e.g., for displaying LTI model variables [m]
z1d_delta = 1e3
z1d = np.linspace(zBot,zTop,np.int(np.rint(1+(zTop-zBot)/z1d_delta)))
x1d_delta = 1e4
x1d = np.linspace(xLef,xRig,np.int(np.rint(1+(xRig-xLef)/x1d_delta)))
x2d,z2d = np.meshgrid(x1d,z1d)

#.. Color palettes for vertical profiles
cGrdM = ['blue','green']
cGrdQ = ['lightblue','lightgreen']

def PltBox(xzLL,xzUR,**kwargs):
    '''
    Plot rectangular box
        xzLL: coordinates (x,z) of lower left corner
        xzUR: coordinates (x,z) of upper right corner
    '''
    plt.plot([xzLL[0],xzLL[0]],[xzLL[1],xzUR[1]],**kwargs)
    plt.plot([xzUR[0],xzUR[0]],[xzLL[1],xzUR[1]],**kwargs)
    plt.plot([xzLL[0],xzUR[0]],[xzLL[1],xzLL[1]],**kwargs)
    plt.plot([xzLL[0],xzUR[0]],[xzUR[1],xzUR[1]],**kwargs)
    return 'PltBox() successful.'

#############################################
### (8) Length scales expressed in kilometers 
#############################################

#.. LTI region boundaries [km]
zBot_km = zBot/1e3
zTop_km = zTop/1e3
xLef_km = xLef/1e3
xRig_km = xRig/1e3

#.. Satellite orbit coordinates [km]
xSatA_km = xSatA/1e3
zSatA_km = zSatA/1e3
xSatB_km = xSatB/1e3
zSatB_km = zSatB/1e3

#.. Default arrays for plotting [km]
z1d_km = z1d/1e3
x1d_km = x1d/1e3
x2d_km = x2d/1e3
z2d_km = z2d/1e3

#################################
### End of file DIPContBas.py ###
#################################
