'''
Daedalus Ionospheric Profiles Continuation (DIPCont) Project
DIPContEst.py: DIPCont parameter estimation and extrapolation.
PYTHON CODE UNDER DEVELOPMENT, PROVIDED AS IS, NO WARRANTY.
Written by Joachim Vogt, Jacobs University Bremen (JUB).
Tested also by Octav Marghitu, ISS Bucharest (ISS), 
Adrian Blagau (ISS, JUB), Leonie Pick (DLR Neustrelitz, JUB), 
and Nele Stachlys (U Potsdam, JUB); in collaboration with 
Stephan Buchert, Swedish Institute of Space Physics in Uppsala, 
Theodoros Sarris, Democritus University of Thrace in Xanthi 
(DUTH), Stelios Tourgaidis (DUTH), Thanasis Balafoutis (DUTH), 
Dimitrios Baloukidis (DUTH), and Panagiotis Pirnaris (DUTH).
This version: v0.2, 2022-07-06.
'''

#.. General imports
import numpy as np
from numpy.random import rand, randn
import matplotlib.pyplot as plt
import pandas as pd
from scipy.optimize import curve_fit
from scipy.stats import linregress

############################################################
### (1) Parametric functions for estimates from in situ data
############################################################

def TnModzLoc(z,Tn0):
    '''
    Neutral temperature Tn (local representation)
    Input: 
        z        : altitude [m]
        Tn0      : neutral temperature at z0 [K]
    To be provided through DIPContBas:
        z0       : reference altitude [m]
        zBot     : altitude of lower boundary [m]
        TnBotEst : neutral temperature estimate at zBot [K]
    Output:
        Tn       : altitude profile of neutral temperature [K]
    '''
    from DIPContBas import z0,zBot,TnBotEst
    Ln0 = Tn0*(z0-zBot)/(Tn0-TnBotEst)
    Tn = Tn0*( 1 + (z-z0)/Ln0 )
    return Tn

def LogTnModzLoc(*args,**kwargs):
    '''
    Logarithm of neutral temperature Tn (local representation)
    as obtained from TnModzLoc()
    '''
    return np.log(TnModzLoc(*args,**kwargs))

def NnModzLoc(z,Nn0):
    '''
    Neutral density Nn (local representation)
    Input: 
        z     : altitude [m]
        Nn0   : neutral density at z0 [K]
    To be provided through DIPContBas:
        z0    : reference altitude [m]
        HNn0  : neutral density scale height at z0 [m]
        IGHNn : inverse gradient of neutral density scale height
    Output:
        Nn    : altitude profile of neutral density [1/m^3]
    '''
    from DIPContBas import z0,HNn0,IGHNn
    return Nn0*np.exp(-IGHNn*np.log(1+(z-z0)/(IGHNn*HNn0)))

def LogNnModzLoc(*args,**kwargs):
    '''
    Logarithm of neutral density Nn (local representation)
    as obtained from NnModzLoc()
    '''
    return np.log(NnModzLoc(*args,**kwargs))

def NeModzLoc(z,Ne0,Lr0cc,NeF=0):
    '''
    Electron density Ne (local representation)
    Input: 
        z     : altitude [m]
        Ne0   : neutral density at z0 [K]
        Lr0cc : radiation absorption length * cos(chi) [m]
        NeF   : F layer contribution to electron density [1/m^3]
    To be provided through DIPContBas:
        z0    : reference altitude [m]
        HNn0  : neutral density scale height at z0 [m]
        IGHNn : inverse gradient of neutral density scale height
    Output:
        Ne    : altitude profile of electron density [1/m^3]
    '''
    from DIPContBas import z0,HNn0,IGHNn
    theta0 = (IGHNn-1)*np.log(1+(z-z0)/(IGHNn*HNn0))
    LogNe = np.log(Ne0) + 0.5*IGHNn/(IGHNn-1)*( 
            - theta0 + HNn0/Lr0cc*(1-np.exp(-theta0)) ) 
    return np.exp(LogNe)+NeF

def LogNeModzLoc(*args,**kwargs):
    '''
    Logarithm of electron density Ne (local representation)
    as obtained from NeModzLoc()
    '''
    return np.log(NeModzLoc(*args,**kwargs))

def TiModzLoc(z,Ti0):
    '''
    Ion temperature Ti (local representation)
    Input: 
        z        : altitude [m]
        Ti0      : ion temperature at z0 [K]
    To be provided through DIPContBas:
        z0       : reference altitude [m]
        zBot     : altitude of lower boundary [m]
        TiBotEst : ion temperature estimate at zBot [K]
    Output:
        Ti       : altitude profile of ion temperature [K]
    '''
    from DIPContBas import z0,zBot,TiBotEst
    Li0 = Ti0*(z0-zBot)/(Ti0-TiBotEst)
    Ti = Ti0*( 1 + (z-z0)/Li0 )
    return Ti

def LogTiModzLoc(*args,**kwargs):
    '''
    Logarithm of ion temperature Ti (local representation)
    as obtained from TiModzLoc()
    '''
    return np.log(TiModzLoc(*args,**kwargs))

def ScaleHeightParLoc(PS,Ts0,TsBot):
    '''
    Compute density scale height parameters from temperature data 
    (local representation, values at z0 and zBot).
    Input:
        PS         : particle species - 'n' or 'i'
        Ts0        : neutral/ion temperature at z0 [K]
        TsBot      : neutral/ion temperature at zBot [K]
    To be provided through DIPContBas:
        z0         : reference altitude [m]
        zBot       : altitude of lower boundary [m]
        IdGasConst : ideal gas constant [kg.m^2/s^2/K/mol]
        GravEarth  : Earth's gravitational acceleration [m/s^2]
        MsLTI      : molar mass of neutrals/ions [kg/mol]
    Output:
        IGHNs      : inverse gradient of density scale height
        HNs0       : density scale height at z0 [m]
    '''
    from  DIPContBas import z0,zBot,IdGasConst,GravEarth
    if PS=='n':
        from  DIPContBas import MnLTI as MsLTI
    elif PS=='i':
        from  DIPContBas import MiLTI as MsLTI
    else:
        print('ScaleHeightParLoc() : unsupported value of PS' )
        MsLTI = np.nan
    HPsBot = IdGasConst*TsBot/MsLTI/GravEarth
    HPs0 = IdGasConst*Ts0/MsLTI/GravEarth
    GradHPs = (HPs0-HPsBot)/(z0-zBot)
    IGHNs = 1 + (z0-zBot)/(HPs0-HPsBot)
    HNs0 = HPs0/(1+GradHPs)
    return IGHNs,HNs0

###############################################
### (2) Models for values at the lower boundary
###############################################

def TnBotPrdx(x):
    '''
    Provide local (x) prediction of parameter TnBot.
    Input:
        x        : local x coordinate [m]
    To be provided through DIPContBas:
        xLef     : x coordinate of left boundary [m]
        xRig     : x coordinate of right boundary [m]
        TnBotLef : TnBot at xLef
        TnBotRig : TnBot at xRig
    Output:
        TnBotPrd : local prediction of TnBot [K]
    '''
    from DIPContBas import xLef,xRig,TnBotLef,TnBotRig
    return TnBotLef + (x-xLef)*(TnBotRig-TnBotLef)/(xRig-xLef)

def TiBotPrdx(x):
    '''
    Provide local (x) prediction of parameter TiBot.
    Input:
        x        : local x coordinate [m]
    To be provided through DIPContBas:
        xLef     : x coordinate of left boundary [m]
        xRig     : x coordinate of right boundary [m]
        TiBotLef : TiBot at xLef
        TiBotRig : TiBot at xRig
    Output:
        TiBotPrd : local prediction of TiBot [K]
    '''
    from DIPContBas import xLef,xRig,TiBotLef,TiBotRig
    return TiBotLef + (x-xLef)*(TiBotRig-TiBotLef)/(xRig-xLef)

#################################################
### (3) Monte Carlo parameter ensemble generation
#################################################

def ParEstMC(dfPrdWin,LenSim=100,EstNeF=False,LinEstLogNe=False):
    '''
    DIPCont Monte Carlo parameter estimation using a set of 
    (windowed) predictions around a single horizontal grid point.
    Input:
        dfPrdWin    : pandas dataframe with windowed predictions
        LenSim      : number of Monte Carlo simulations
        EstNeF      : if True, estimate also the F layer contribution
        LinEstLogNe : use linear estimator for LogNe parameters
    To be provided through DIPContBas:
        *Init       : initial guesses for parameter estimates
    To be provided through model functions:
        TnBotPrd    : local prediction of Tn at zBot [K]
        TiBotPrd    : local prediction of Ti at zBot [K] 
    DIPContBas variables that are modified in this function:
        z0,Tn0,IGHNn,HNn0,Nn0,Ne0,Lr0cc,NeF,Ti0,IGHNi,HNi0
    Output:
        dfPar       : pandas dataframe with parameter estimates
    '''
    import DIPContBas as DCB
    #.. initial guesses for parameter estimation
    Tn0Init = DCB.Tn0Init
    Nn0Init = DCB.Nn0Init
    Ne0Init = DCB.Ne0Init
    Lr0ccInit = DCB.Lr0ccInit
    if EstNeF: NeFInit = DCB.NeFInit
    Ti0Init = DCB.Ti0Init
    #.. initialize arrays to store parameter ensembles
    ParStr = ['z0','Tn0','IGHNn','HNn0','Nn0','Ne0','Lr0cc','NeF','Ti0','IGHNi','HNi0']
    dfPar = pd.DataFrame( np.full((LenSim,len(ParStr)),np.nan),\
                              columns=ParStr )
    z = dfPrdWin['z']
    for iSim in range(LenSim):
        #.. set reference altitude
        DCB.z0 = np.min(z)
        dfPar.loc[iSim,'z0'] = DCB.z0
        #.. add noise to predictions at lower boundary
        DCB.TnBotEst = DCB.TnBotPrd + DCB.RelErrTnBot*randn(1)
        DCB.TiBotEst = DCB.TiBotPrd + DCB.RelErrTiBot*randn(1)
        #.. add noise to predictions along satellite orbits
        LogTnDat = dfPrdWin['LogTn'] + DCB.RelErrTn*randn(z.size)
        LogNnDat = dfPrdWin['LogNn'] + DCB.RelErrNn*randn(z.size)
        LogNeDat = dfPrdWin['LogNe'] + DCB.RelErrNe*randn(z.size)
        LogTiDat = dfPrdWin['LogTi'] + DCB.RelErrTi*randn(z.size)
        #.. estimate local neutral temperature amplitude
        pEst = curve_fit(LogTnModzLoc,z,LogTnDat,p0=[Tn0Init])
        DCB.Tn0 = pEst[0][0]
        dfPar.loc[iSim,'Tn0'] = DCB.Tn0
        #.. determine neutral density scale height parameters
        DCB.IGHNn,DCB.HNn0 = ScaleHeightParLoc('n',DCB.Tn0,DCB.TnBotEst)
        dfPar.loc[iSim,'IGHNn'] = DCB.IGHNn
        dfPar.loc[iSim,'HNn0'] = DCB.HNn0
        #.. estimate local neutral density amplitude
        pEst = curve_fit(LogNnModzLoc,z,LogNnDat,p0=[Nn0Init])
        DCB.Nn0 = pEst[0][0]
        dfPar.loc[iSim,'Nn0'] = DCB.Nn0
        #.. estimate local electron density parameters
        if EstNeF:
            p0array = [Ne0Init,Lr0ccInit,NeFInit]
            pEst = curve_fit(LogNeModzLoc,z,LogNeDat,p0=p0array)
            pEst0 = pEst[0][:]
            DCB.NeF = pEst0[2]
            dfPar.loc[iSim,'NeF'] = DCB.NeF
        else:
            p0array = [Ne0Init,Lr0ccInit]
            DCB.NeF = 0
            dfPar.loc[iSim,'NeF'] = DCB.NeF
            if LinEstLogNe:
                LogNnPrdLoc = LogNnModzLoc(z,DCB.Nn0)
                LogNeDatRed = LogNeDat - 0.5*LogNnPrdLoc
                theta0 = (DCB.IGHNn-1)*np.log(1+(z-DCB.z0)/DCB.IGHNn/DCB.HNn0)
                sl,ic,rv,pv,se = linregress(1-np.exp(-theta0),LogNeDatRed)
                pEst0 = [np.exp(ic+0.5*np.log(DCB.Nn0)),\
                             0.5*DCB.IGHNn/(DCB.IGHNn-1)*DCB.HNn0/sl]
            else:
                pEst = curve_fit(LogNeModzLoc,z,LogNeDat,p0=p0array)
                pEst0 = pEst[0][:]
        DCB.Ne0 = pEst0[0]
        dfPar.loc[iSim,'Ne0'] = DCB.Ne0
        DCB.Lr0cc = pEst0[1]
        dfPar.loc[iSim,'Lr0cc'] = DCB.Lr0cc
        #.. estimte local ion temperature amplitude
        pEst = curve_fit(LogTiModzLoc,z,LogTiDat,p0=[Ti0Init])
        DCB.Ti0 = pEst[0][0]
        dfPar.loc[iSim,'Ti0'] = DCB.Ti0
        #.. determine ion density scale height parameters
        DCB.IGHNi,DCB.HNi0 = ScaleHeightParLoc('i',DCB.Ti0,DCB.TiBotEst)
        dfPar.loc[iSim,'IGHNi'] = DCB.IGHNi
        dfPar.loc[iSim,'HNi0'] = DCB.HNi0
    return dfPar

def GrdParEstMC(dfPrd,LenSim=100,EstNeF=False,LinEstLogNe=False):
    '''
    DIPCont Monte Carlo parameter estimation using model predictions
    along the entire satellite orbit(s), and for sets of measurements 
    in windows (xWin) around the horizontal positions of the array xGrd.
    Input:
        dfPrd       : pandas dataframe with predictions
        LenSim      : number of Monte Carlo simulations
        EstNeF      : if True, estimate also the F layer contribution
        LinEstLogNe : if True, use linear estimator for LogNe parameters
    To be provided through DIPContBas:
        xGrd        : grid of horizontal (x) positions [m]
        xWin        : horizontal window for data selection [m]
    DIPContBas variables that are modified in this function:
        TnBotPrd,TiBotPrd
    Output:
        dfGrdPar    : pandas dataframe with parameter estimates
    '''
    import DIPContBas as DCB
    xGrd = DCB.xGrd
    xWin = DCB.xWin
    #.. initialize dfGrdPar
    dfGrdPar = {}
    for iGrd in range(xGrd.size):
        #.. initialize Monte Carlo simulations
        indWin = np.abs(dfPrd['x']-xGrd[iGrd]) <= xWin/2
        DCB.TnBotPrd = TnBotPrdx(xGrd[iGrd])
        DCB.TiBotPrd = TiBotPrdx(xGrd[iGrd])
        #.. run Monte Carlo simulations
        dfGrdPar[iGrd] = ParEstMC(dfPrd[indWin],LenSim=LenSim,\
                             EstNeF=EstNeF,LinEstLogNe=LinEstLogNe)
    #.. append Monte Carlo results to dfGrdPar
    dfGrdPar = pd.concat(dfGrdPar,names=['iGrd','iSim'])
    return dfGrdPar

########################################################
### (4) Local-regional conversion of Ne model parameters
########################################################

def NeParCnvReg2Loc(NeIpk,zIpk,HNnIpk,IGHNn,z0):
    '''
    Convert Ne peak parameters to Ne local parameters.
    Input:
        NeIpk  : electron density peak value [1/m^3]
        zIpk   : electron density peak altitude [m]
        HNnIpk : neutral density scale height at zIpk [m]
        IGHNn  : inverse gradient of density scale height [m]
        z0     : reference altitude [m]
    Output:
        Ne0    : electron density at z0 [1/m^3]
        Lr0cc  : radiation absorption length * cos(chi) [m]
        HNn0   : neutral density scale height at z0 [m]
    '''
    zetaIpk = (z0-zIpk)/HNnIpk
    thetaIpk = (IGHNn-1)*np.log(1+zetaIpk/IGHNn)
    Ne0 = NeIpk*np.exp( 0.5*IGHNn/(IGHNn-1)*(
            -thetaIpk + 1 -np.exp(-thetaIpk) ) )
    HNn0 = HNnIpk*(1+(z0-zIpk)/IGHNn/HNnIpk)
    zeta0 = (zIpk-z0)/HNn0
    theta0 = (IGHNn-1)*np.log(1+zeta0/IGHNn)
    Lr0cc = HNn0*np.exp(-theta0)
    return Ne0,Lr0cc,HNn0

def NeParCnvLoc2Reg(Ne0,z0,HNn0,IGHNn,Lr0cc):
    '''
    Convert Ne local parameters to Ne peak parameters.
    Input:
        Ne0    : electron density at z0 [1/m^3]
        z0     : reference altitude [m]
        HNn0   : neutral density scale height at z0 [m]
        IGHNn  : inverse gradient of density scale height [m]
        Lr0cc  : radiation absorption length * cos(chi) [m]
    Output:
        NeIpk  : electron density peak value [1/m^3]
        zIpk   : electron density peak altitude [m]
        HNnIpk : neutral density scale height at zIpk [m]
    '''
    HoL = HNn0/Lr0cc
    NeIpk = Ne0*np.exp( 0.5*IGHNn/(IGHNn-1)*(-np.log(HoL)+HoL-1) )
    zIpk = z0 + HNn0*IGHNn*( HoL**(1/(IGHNn-1)) - 1 )
    HNnIpk = HNn0*(1+(zIpk-z0)/IGHNn/HNn0)
    return NeIpk,zIpk,HNnIpk

############################################
### (5) Extrapolation (profile continuation)
############################################

def VarExtMC(z,dfParSim):
    '''
    DIPCont extrapolation of variables along altitude z using 
    a single set of model parameters (at one horizontal position).
    Input:
        z        : one-dimensional array of altitudes [m] 
        dfParSim : pandas dataframe with estimated parameters
    DIPContBas variables that are modified in this function:
        z0,Tn0,IGHNn,HNn0,Nn0,Ne0,Lr0cc,NeF,Ti0,IGHNi,HNi0
    Output:
        dfExtSim : altitude profiles of extrapolated variables
    '''
    import DIPContBas as DCB
    #.. update DCB parameters
    DCB.z0 = dfParSim['z0'].values
    DCB.Tn0 = dfParSim['Tn0'].values
    DCB.HNn0 = dfParSim['HNn0'].values
    DCB.IGHNn = dfParSim['IGHNn'].values
    DCB.Nn0 = dfParSim['Nn0'].values
    DCB.Ne0 = dfParSim['Ne0'].values
    DCB.Lr0cc = dfParSim['Lr0cc'].values
    DCB.NeF = dfParSim['NeF'].values
    DCB.Ti0 = dfParSim['Ti0'].values
    DCB.HNi0 = dfParSim['HNi0'].values
    DCB.IGHNi = dfParSim['IGHNi'].values
    #.. compute altitude profiles using DCB parameters 
    TnExt = DCB.Tn0*( 1 + (z-DCB.z0)/DCB.IGHNn/DCB.HNn0 )
    NnExt = NnModzLoc(z,DCB.Nn0)
    NeExt = NeModzLoc(z,DCB.Ne0,DCB.Lr0cc,NeF=DCB.NeF)
    TiExt = DCB.Ti0*( 1 + (z-DCB.z0)/DCB.IGHNi/DCB.HNi0 )
    FCinExt = DCB.SCin*NnExt*np.sqrt(DCB.IdGasConst*TiExt/DCB.MiLTI)
    FCinOverFGi = FCinExt/DCB.FGiLTI
    CpedExt = NeExt*DCB.ElemCharge/DCB.BmLTI*FCinOverFGi/(1+FCinOverFGi**2)
    iGrdExt = np.full(z.size,dfParSim.index[0][0],dtype=int)
    iSimExt = np.full(z.size,dfParSim.index[0][1],dtype=int)
    indexExt = pd.MultiIndex.from_arrays([iGrdExt,iSimExt,],\
										 names=['iGrd','iSim'])
    ExtStr = ['z','Tn','Nn','Ne','Ti','FCin','Cped']
    dfExtSim = pd.DataFrame(np.stack((z,TnExt,NnExt,NeExt,TiExt,\
            FCinExt,CpedExt),axis=-1),columns=ExtStr,index=indexExt)
    return dfExtSim

def GrdVarExtMC(z,dfGrdPar):
    '''
    DIPCont extrapolation of variables along altitude z for 
    all horizontal positions of the array xGrd.
    Input:
        z        : one-dimensional array of altitudes [m] 
        dfGrdPar : pandas dataframe with estimated parameters
    Output:
        dfGrdExt : altitude profiles of extrapolated variables
    '''
    dfGrdExt = {}
    for iGrdSim in range(len(dfGrdPar)):
        dfPar = dfGrdPar.iloc[[iGrdSim]]
        dfGrdExt[iGrdSim] = VarExtMC(z,dfPar)
    dfGrdExt = pd.concat(dfGrdExt).droplevel(level=0)
    return dfGrdExt

def RelErrVarMC(VarEst,VarPrd=None,ErrTyp='RMS'):
    '''
    Compute relative error of an extrapolated variable as 
    obtained from Monte Carlo runs.
    Input:
        VarEst   : estimated variables, array of shape 
                    (z1d.size,LenSim) for the one-dim case,
                    (z1d.size,xGrd.size,LenSim) for the two-dim case,
                      such as the result of dfGrdExt[Var].values\
                      .reshape((xGrd.size,LenSim,z1d.size)).transpose(2,0,1)
        VarPrd   : predictions, array of shape
                    (z1d.size,) for the one-dim case,
                    (z1d.size,xGrd.size) for the two-dim case.
                  If None, then VarPrd is set to the median of VarEst.
        ErrTyp  : 'RMS' - root mean square deviation from prediction,
                  'AAD' - average absolute deviation from prediction,
                  'AD5' - 0.5 quantile (median) of abs dev from pred,
                  'AD9' - 0.9 quantile of abs dev from pred,
                  'QND' - quartile-based normalized dev from pred.
    Output:
        RelErr  : relative error, array of the same shape as VarPrd.
    '''
    #.. if VarPrd is not provided, assign median of estimates
    if VarPrd is None:
        VarPrd = np.median(VarEst,axis=-1)
    #.. compute deviation of estimates from prediction 
    VarDev = VarEst - VarPrd[...,None]
    #.. depending on ErrTyp, compute absolute error
    if ErrTyp=='RMS':
        AbsErr = np.sqrt(np.mean(VarDev**2,axis=-1))
    elif ErrTyp=='AAD':
        AbsErr = np.mean(np.abs(VarDev),axis=-1)
    elif ErrTyp=='AD5':
        AbsErr = np.median(np.abs(VarDev),axis=-1)  
    elif ErrTyp=='AD9':
        AbsErr = np.quantile(np.abs(VarDev),0.9,axis=-1)
    elif ErrTyp=='QND':
        from scipy.stats import norm
        NF = 1/(norm.ppf(0.75)-norm.ppf(0.25))**2
        VarQnt = np.quantile(VarEst,[0.5,0.25,0.75],axis=-1)
        AbsErr = np.sqrt( (VarQnt[0,:]-VarPrd)**2 \
                              + NF*(VarQnt[2,:]-VarQnt[1,:])**2 )
    else:
        print('RelErrVarMCzs : non-supported value of ErrTyp')
        AbsErr = np.nan
    #.. normalize absolute error to obtain relative error
    RelErr = AbsErr/VarPrd
    return RelErr

#######################################################
### (6) Plot functions for estimation and extrapolation
#######################################################

def PltQnt1d(Var,VarEst,VarPrd=None,zA=None,zB=None,\
                 zLabel=True,cM='gray',cQ='lightgray'):
    '''
    Plot quantiles (percentiles) of an extrapolated variable as 
    obtained from Monte Carlo runs - for one-dim applications.
    Input:
        Var     : variable of interest [string]
        VarEst  : estimated variables, shape (z1d.size,LenSim)
        VarPrd  : prediction vector of shape (z1d.size,)
        zA      : if not None, plot horizontal line at zA
        zB      : if not None, plot horizontal line at zB
        zLabel  : if False, omit label at vertical (z) axis
        cM      : color used for median, quartiles, and prediction
        cQ      : color used for (0.05,0.95) quantiles
    To be provided through DIPContBas: auxiliary plot parameters.
    '''
    from DIPContBas import z1d,z1d_km
    from DIPContBas import zBot_km,zTop_km
    #.. compute array of quantiles (quantile dimension: axis=0)
    VarQnt = np.quantile(VarEst,[0.5,0.25,0.75,0.05,0.95],axis=-1)
    #.. depending on Var, scale the quantile array
    if Var=='Tn':
        VarQntPlt = VarQnt
        plt.title(r'$T_n$ percentiles')
        plt.xlabel(r'$T_n~[\mathrm{K}]$')
    elif Var=='Nn':
        VarQntPlt = np.log10(VarQnt)
        plt.title(r'$N_n$ percentiles')
        plt.xlabel(r'$\log_{10} ( N_n~[\mathrm{m}^{-3}] )$')
    elif Var=='Ne':
        VarQntPlt = VarQnt
        plt.title(r'$N_e$ percentiles')
        plt.xlabel(r'$N_e~[\mathrm{m}^{-3}]$')
    elif Var=='Ti':
        VarQntPlt = VarQnt
        plt.title(r'$T_i$ percentiles')
        plt.xlabel(r'$T_i~[\mathrm{K}]$')
    elif Var=='FCin':
        VarQntPlt = np.log10(VarQnt)
        plt.title(r'$\nu_{{in}}$ percentiles')
        plt.xlabel(r'$\log_{10} ( \nu_{in}~[\mathrm{Hz}] )$')
    elif Var=='Cped':
        VarQntPlt = 1e6*VarQnt
        plt.title(r'$\sigma_\mathrm{{P}}$ percentiles')
        plt.xlabel(r'$\sigma_\mathrm{P}~[\mu\mathrm{S/m}]$')
    else:
        print('PltQnt1d : non-supported value of Var')
    #.. depending on Var, scale the prediction
    if VarPrd is not None:
        if Var in ['Tn','Ne','Ti']:
            VarPrd = VarPrd
        elif Var in ['Nn','FCin']:
            VarPrd = np.log10(VarPrd)
        elif Var=='Cped':
            VarPrd = 1e6*VarPrd
        else:
            print('PltQnt1d : non-supported value of Var')
        #.. line plot of the prediction
        plt.plot(VarPrd,z1d_km,color=cM,linewidth=2,\
                 label='Model prediction')
    #.. line plot of the median
    plt.plot(VarQntPlt[0,:],z1d_km,color=cM,linewidth=2,\
                 linestyle='--',label='Median estimate')
    #.. line plots of the 25% and 75% percentiles (quartiles)
    plt.plot(VarQntPlt[1,:],z1d_km,color=cM,linewidth=2,\
                 linestyle=':',label='Percentiles (25,75)')
    plt.plot(VarQntPlt[2,:],z1d_km,color=cM,linewidth=2,linestyle=':')
    #.. shaded region between the 5% and 95% percentiles
    plt.fill_betweenx(z1d_km,VarQntPlt[3,:],VarQntPlt[4,:],\
                               color=cQ,label='Percentiles (5,95)')
    plt.autoscale(enable=True,axis='x',tight=True)
    #.. plot horizontal lines at zA and zB
    if zA is not None:
        VarMin,VarMax = plt.xlim()
        plt.plot([VarMin,VarMax],[zA/1e3,zA/1e3],c='k',ls='--',lw=2)
    if zB is not None:
        VarMin,VarMax = plt.xlim()
        plt.plot([VarMin,VarMax],[zB/1e3,zB/1e3],c='k',ls='--',lw=2)
    #.. grid, legend, labels, limits
    plt.grid()
    plt.legend()
    if zLabel:
        plt.ylabel(r'Altitude $z$ [km]')
    else:
        ax = plt.gca()
        ax.axes.yaxis.set_ticklabels([])
    plt.ylim([zBot_km,zTop_km])
    return 'PltQnt1d() successful.'

def PltHor1d(Var,VarEst,VarPrd=None,ErrTyp='RMS',zA=None,zB=None,\
                 zLabel=True,cM='gray'):
    '''
    Plot Monte Carlo extrapolation horizons (one-dim case).
    Input:
        Var     : variable of interest [string]
        VarEst  : estimated variables, shape (z1d.size,LenSim)
        VarPrd  : prediction vector of shape (z1d.size,)
        ErrTyp  : see RelErrVarMC()
        zA      : if not None, plot horizontal line at zA
        zB      : if not None, plot horizontal line at zB
        zLabel  : if False, omit label at vertical (z) axis
        cM      : color used for median, quartiles, and prediction
    To be provided through DIPContBas: auxiliary plot parameters.
    '''
    from DIPContBas import z1d,z1d_km
    from DIPContBas import zBot_km,zTop_km
    from DIPContBas import ExtHorPct as EHP
    from DIPContBas import ExtHorCol as EHC
    from DIPContBas import ExtHorLst as EHL
    #.. compute relative errors of variable estimates
    RelErrPct = 100*RelErrVarMC(VarEst,VarPrd=VarPrd,ErrTyp=ErrTyp)
    RelErrPct_Min = np.min(RelErrPct)
    RelErrPct_Down = RelErrPct[:np.argmin(RelErrPct)]
    #.. x axis limits
    pMin = 0.1
    pMax = 100
    plt.xlim([pMin,pMax])
    plt.xscale('log')
    plt.xticks(EHP[1::2],['{:.0f}'.format(p) for p in EHP[1::2]])
    #.. plot title and x axis labels
    if Var=='Tn':
        plt.title(r'$T_n$ extrapolation horizons')
        plt.xlabel(r'$\delta{T_n}/T_n$ [%]')
    elif Var=='Nn':
        plt.title(r'$N_n$ extrapolation horizons')
        plt.xlabel(r'$\delta{N_n}/N_n$ [%]')
    elif Var=='Ne':
        plt.title(r'$N_e$ extrapolation horizons')
        plt.xlabel(r'$\delta{N_e}/N_e$ [%]')
    elif Var=='Ti':
        plt.title(r'$T_i$ extrapolation horizons')
        plt.xlabel(r'$\delta{T_i}/T_i$ [%]')
    elif Var=='FCin':
        plt.title(r'$\nu_{{in}}$ extrapolation horizons')
        plt.xlabel(r'$\delta\nu_{in}/\nu_{in}$ [%]')
    elif Var=='Cped':
        plt.title(r'$\sigma_\mathrm{{P}}$ extrapolation horizons')
        plt.xlabel(r'$\delta\sigma_\mathrm{P}/\sigma_\mathrm{P}$ [%]')
    else:
        print('PltHor1d : non-supported value of Var')
    #.. plot relative error as function of altitude
    plt.plot(RelErrPct,z1d_km,color=cM,linewidth=2)
    #.. plot percentage thresholds as vertical lines, and
    #.. extrpolation horizons as horizontal lines 
    for iHor in range(len(EHP)):
        pHor = EHP[iHor]
        cHor = EHC[iHor]
        lHor = EHL[iHor]
        plt.plot([pHor,pHor],[zBot_km,zTop_km],c=cHor,ls=lHor,lw=2)
        if pHor>=RelErrPct_Min:
            zHor_km = z1d_km[np.argmin(np.abs(RelErrPct_Down-pHor))]
            plt.plot([pMin,pHor],[zHor_km,zHor_km],c=cHor,ls=lHor,lw=2)
    #.. plot horizontal lines at zA and zB    
    if zA is not None:
        plt.plot([pMin,pMax],[zA/1e3,zA/1e3],c='k',ls='--',lw=2)
    if zB is not None:
        plt.plot([pMin,pMax],[zB/1e3,zB/1e3],c='k',ls='--',lw=2)
    #.. grid, labels, limits
    plt.grid()
    if zLabel:
        plt.ylabel(r'Altitude $z$ [km]')
    else:
        ax = plt.gca()
        ax.axes.yaxis.set_ticklabels([])
    plt.ylim([zBot_km,zTop_km])
    return 'PltHor1d() successful.'

def fmtPct(Pct):
    '''
    Compact formating of number Pct with up to three decimals.
    '''
    PctStr = f'{Pct:.3f}'
    if PctStr.endswith('000'):
        PctStr = f'{Pct:.0f}'
    elif PctStr.endswith('00'):
        PctStr = f'{Pct:.1f}'
    elif PctStr.endswith('0'):
        PctStr = f'{Pct:.2f}'
    if plt.rcParams['text.usetex']:
        PctStr = rf'{PctStr} \%'
    else:
        PctStr = f'{PctStr} %'
    return PctStr

def PltHor2d(VarEst,VarPrd=None,ErrTyp='RMS'):
    '''
    Plot Monte Carlo extrapolation horizons (two-dim case) as 
    contours into the plot window created by DIPContMod.PltMod2d().
    Input:
        VarEst  : estimated variables, shape (z1d.size,xGrd.size,LenSim)
                    such as the result of dfGrdExt[Var].values\
                    .reshape((xGrd.size,LenSim,z1d.size)).transpose(2,0,1)
        VarPrd  : predictions, shape (z1d.size,xGrd.size).
                  If None, then VarPrd is set to the median of VarEst.
        ErrTyp  : see RelErrVarMC()
    To be provided through DIPContBas: auxiliary plot parameters, and
        z1d     : profile of vertical distances [m]
        xGrd    : grid of horizontal distances [m]
    '''
    from DIPContBas import z1d,xGrd
    from DIPContBas import ExtHorPct as EHP
    from DIPContBas import ExtHorCol as EHC
    from DIPContBas import ExtHorLst as EHL
    RelErrPct = 100*RelErrVarMC(VarEst,VarPrd=VarPrd,ErrTyp=ErrTyp)
    CS = plt.contour(xGrd/1e3,z1d/1e3,RelErrPct,EHP,\
                         colors=EHC,linestyles=EHL,linewidths=2)
    plt.clabel(CS,EHP[1::2],inline=True,fmt=fmtPct,fontsize=10)
    return 'PltHor2d() successful.'

#################################
### End of file DIPContEst.py ###
#################################
