'''
Daedalus Ionospheric Profiles Continuation (DIPCont) Project
DIPContMod.py: DIPCont parametric models of LTI region variables.
PYTHON CODE UNDER DEVELOPMENT, PROVIDED AS IS, NO WARRANTY.
Written by Joachim Vogt, Jacobs University Bremen (JUB).
Tested also by Octav Marghitu, ISS Bucharest (ISS), 
Adrian Blagau (ISS, JUB), Leonie Pick (DLR Neustrelitz, JUB), 
and Nele Stachlys (U Potsdam, JUB); in collaboration with 
Stephan Buchert, Swedish Institute of Space Physics in Uppsala, 
Theodoros Sarris, Democritus University of Thrace in Xanthi 
(DUTH), Stelios Tourgaidis (DUTH), Thanasis Balafoutis (DUTH), 
Dimitrios Baloukidis (DUTH), and Panagiotis Pirnaris (DUTH).
This version: v0.2, 2022-07-06.
'''

#.. General imports
import numpy as np
import matplotlib.pyplot as plt
import pandas as pd

##################################################################
### (1) Horizontal profiles of boundary values and peak parameters
##################################################################

def VarBotModx(Var,x):
    '''
    Provide horizontal (x) profile of a variable at zBot.
    Input:
        Var    : variable of interest [string]
        x      : horizontal distance(s) [float]
    To be provided through DIPContBas:
        xLef   : x coordinate of left boundary [m]
        xRig   : x coordinate of right boundary [m]
        dfVarBotLef : dataframe with 'Bot' variables at xLef
        dfVarBotRig : dataframe with 'Bot' variables at xRig
    Output:
        VarBot : horizontal (x) profile of variable Var at zBot 
    '''
    from DIPContBas import xLef,xRig,dfVarBotLef,dfVarBotRig
    if Var in ['Tn','Nn','Ti']:
        VarBotLef = dfVarBotLef[Var]
        VarBotRig = dfVarBotRig[Var]
        VarBot = VarBotLef + (x-xLef)*(VarBotRig-VarBotLef)/(xRig-xLef)
    else:
        print('VarBotModx() : unsupported value of Var.' )
        VarBot = np.nan
    return VarBot

def VarTopModx(Var,x):
    '''
    Provide horizontal (x) profile of a variable at zTop.
    Input:
        Var    : variable of interest [string]
        x      : horizontal distance(s) [float]
    To be provided through DIPContBas:
        xLef   : x coordinate of left boundary [m]
        xRig   : x coordinate of right boundary [m]
        dfVarTopLef : dataframe with 'Top' variables at xLef
        dfVarTopRig : dataframe with 'Top' variables at xRig
    Output:
        VarTop : horizontal (x) profile of variable Var at zTop 
    '''
    from DIPContBas import xLef,xRig,dfVarTopLef,dfVarTopRig
    if Var in ['Tn','Ti']:
        VarTopLef = dfVarTopLef[Var]
        VarTopRig = dfVarTopRig[Var]
        VarTop = VarTopLef + (x-xLef)*(VarTopRig-VarTopLef)/(xRig-xLef)
    else:
        print('VarTopModx() : unsupported value of Var.' )
        VarTop = np.nan
    return VarTop

def ScaleHeightParReg(PS,TsTop,TsBot):
    '''
    Compute density scale height parameters from temperature data 
    (regional representation, values at zTop and zBot).
    Input:
        PS         : particle species - 'n' or 'i'
        TsTop      : neutral/ion temperature at zTop [K]
        TsBot      : neutral/ion temperature at zBot [K]
    To be provided through DIPContBas:
        zTop       : altitude of upper boundary [m]
        zBot       : altitude of lower boundary [m]
        IdGasConst : ideal gas constant [kg.m^2/s^2/K/mol]
        GravEarth  : Earth's gravitational acceleration [m/s^2]
        MsLTI      : molar mass of neutrals/ions [kg/mol]
    Output:
        IGHNs      : inverse gradient of density scale height
        HNsTop     : density scale height at zTop [m]
        HNsBot     : density scale height at zBot [m]
    '''
    from  DIPContBas import zTop,zBot,IdGasConst,GravEarth
    if PS=='n':
        from  DIPContBas import MnLTI as MsLTI
    elif PS=='i':
        from  DIPContBas import MiLTI as MsLTI
    else:
        print('ScaleHeightParLoc() : unsupported value of PS' )
        MsLTI = np.nan
    HPsBot = IdGasConst*TsBot/MsLTI/GravEarth
    HPsTop = IdGasConst*TsTop/MsLTI/GravEarth
    GradHPs = (HPsTop-HPsBot)/(zTop-zBot)
    IGHNs = 1 + (zTop-zBot)/(HPsTop-HPsBot)
    HNsBot = HPsBot/(1+GradHPs)
    HNsTop = HPsTop/(1+GradHPs)
    return IGHNs,HNsTop,HNsBot

def IpkParAZC(x,NeIpkVal,NeIpkDel,zIpkVal,zIpkDel):
    '''
    Provide horizontal (x) profiles of ionization peak (Ipk) 
    parameters when LTIModelType is 'NeAuroralZoneCrossing'.
    Input:
        x        : horizontal distance(s) [float]
        NeIpkVal : Ne peak density reference value [1/m^3]
        NeIpkDel : Ne peak density variation parameter [1/m^3]
        zIpkVal  : Ne peak altitude reference value [m]
        zIpkDel  : Ne peak altitude variation parameter [m]
    To be provided through DIPContBas:
        xLef   : x coordinate of left boundary [m]
        xRig   : x coordinate of right boundary [m]
    Output:
        zIpk     : horizontal profile of peak altitude [m]
        NeIpk    : horizontal profile of peak density [1/m^3]
    '''
    from DIPContBas import xLef,xRig
    xFunction = 1 + np.cos( 4*np.pi*x/(xRig-xLef) )
    NeIpk = NeIpkVal - 0.5*NeIpkDel*xFunction
    zIpk = zIpkVal + 0.5*zIpkDel*xFunction
    return NeIpk,zIpk

#######################################################
### (2) Prediction models in terms of coordinates (z,x)
#######################################################

def VarModzx(Var,z,x):
    '''
    Provide two-dimensional (z,x) model distribution 
    of a variable in the LTI region.
    Input:
        Var    : variable of interest [string]
        z      : vertical distance(s)  [m]
        x      : horizontal distance(s) [m]
    To be provided through DIPContBas:
        zBot   : altitude of lower boundary [m]
        zTop   : altitude of upper boundary [m]
    To be provided through DIPContBas for Var=='Ne':
        LTIModelType : select Ne horizontal variation mode
        NeF          : F layer contribution to Ne [1/m^3]
        NeIpk*,zIpk* : Ne peak parameters
    To be provided through DIPContBas for Var=='FCin':
        SCin         : ion-neutral collision cross-section [m^2]
        IdGasConst   : ideal gas constant [kg.m^2/s^2/K/mol]
        MiLTI        : molar mass of ions [kg/mol]
    To be provided through DIPContBas for Var=='Cped':
        FGiLTI       : ion (angular) gyrofrequency [rad/s]
        ElemCharge   : elementary charge [C]
        BmLTI        : magnetic field strength [T]
    Output:
        VarMod : two-dimensional (z,x) model distribution of Var
    '''
    from  DIPContBas import zBot,zTop
    if Var in ['Tn','Ti']:
        VarBot = VarBotModx(Var,x)
        VarTop = VarTopModx(Var,x)
        VarMod = VarBot + (z-zBot)*(VarTop-VarBot)/(zTop-zBot)
    elif Var=='Nn':
        TnBot = VarBotModx('Tn',x)
        TnTop = VarTopModx('Tn',x)
        IGHNn,HNnTop,HNnBot = ScaleHeightParReg('n',TnTop,TnBot)
        NnBot = VarBotModx('Nn',x)
        LogNn = np.log(NnBot) - IGHNn*np.log(1+(z-zBot)/IGHNn/HNnBot)
        VarMod = np.exp(LogNn)
    elif Var=='Ne':
        TnBot = VarBotModx('Tn',x)
        TnTop = VarTopModx('Tn',x)
        IGHNn,HNnTop,HNnBot = ScaleHeightParReg('n',TnTop,TnBot)
        from  DIPContBas import LTIModelType,NeF
        if LTIModelType=='NeAuroralZoneCrossing':
            from  DIPContBas import NeIpkVal,NeIpkDel,zIpkVal,zIpkDel
            NeIpk,zIpk = IpkParAZC(x,NeIpkVal,NeIpkDel,zIpkVal,zIpkDel)
        else:
            from  DIPContBas import NeIpk,zIpk
            NeIpk = np.ones(z.shape)*NeIpk
            zIpk = np.ones(z.shape)*zIpk
        HNnIpk = HNnBot*(1+(zIpk-zBot)/IGHNn/HNnBot)
        thetaIpk = (IGHNn-1)*np.log(1+(z-zIpk)/IGHNn/HNnIpk)
        LogNe = np.log(NeIpk) \
                + 0.5*IGHNn/(IGHNn-1)*(-thetaIpk+1-np.exp(-thetaIpk))
        VarMod = np.exp(LogNe)+NeF
    elif Var=='FCin':
        from DIPContBas import SCin,IdGasConst,MiLTI
        NnMod = VarModzx('Nn',z,x)
        TiMod = VarModzx('Ti',z,x)
        VarMod = SCin*NnMod*np.sqrt(IdGasConst*TiMod/MiLTI)
    elif Var=='Cped':
        from DIPContBas import FGiLTI,ElemCharge,BmLTI
        NeMod = VarModzx('Ne',z,x)
        FCinMod = VarModzx('FCin',z,x)
        FCinOverFGi = FCinMod/FGiLTI
        VarMod = NeMod*ElemCharge/BmLTI*FCinOverFGi/(1+FCinOverFGi**2)
    else:
        print('VarModzx() : unsupported value of Var.' )
        VarMod = np.nan
    return VarMod

def SatPrd(TwoSat=True):
    '''
    Provide model predictions along the orbits of both 
    SatA and SatB (TwoSat=True) or SatA only (TwoSat=False).
    Orbital altitudes (zSatA,zSatB) and horizontal distances
    (xSatA,xSatB) are to be provided through DIPContBas.
    The results are returned in a Pandas DataFrame object.
    '''
    from DIPContBas import zSatA,xSatA,tSatA
    LogTnSatAPrd = np.log(VarModzx('Tn',zSatA,xSatA))
    LogNnSatAPrd = np.log(VarModzx('Nn',zSatA,xSatA))
    LogNeSatAPrd = np.log(VarModzx('Ne',zSatA,xSatA))
    LogTiSatAPrd = np.log(VarModzx('Ti',zSatA,xSatA))
    if TwoSat:
        from DIPContBas import zSatB,xSatB,tSatB
        LogTnSatBPrd = np.log(VarModzx('Tn',zSatB,xSatB))
        LogNnSatBPrd = np.log(VarModzx('Nn',zSatB,xSatB))
        LogNeSatBPrd = np.log(VarModzx('Ne',zSatB,xSatB))
        LogTiSatBPrd = np.log(VarModzx('Ti',zSatB,xSatB))
        zPrd = np.concatenate((zSatA,zSatB))
        xPrd = np.concatenate((xSatA,xSatB))
        LogTnPrd = np.concatenate((LogTnSatAPrd,LogTnSatBPrd))
        LogNnPrd = np.concatenate((LogNnSatAPrd,LogNnSatBPrd))
        LogNePrd = np.concatenate((LogNeSatAPrd,LogNeSatBPrd))
        LogTiPrd = np.concatenate((LogTiSatAPrd,LogTiSatBPrd))
    else:
        zPrd = zSatA
        xPrd = xSatA
        LogTnPrd = LogTnSatAPrd
        LogNnPrd = LogNnSatAPrd
        LogNePrd = LogNeSatAPrd
        LogTiPrd = LogTiSatAPrd
    PrdStr = ['z','x','LogTn','LogNn','LogNe','LogTi']
    PrdMat = np.stack((zPrd,xPrd,LogTnPrd,LogNnPrd,LogNePrd,LogTiPrd),\
                          axis=-1)
    dfPrd = pd.DataFrame(PrdMat,columns=PrdStr)
    return dfPrd

################################################################
### (3) Plot functions for individual LTI region model variables
################################################################

def PltMod2d(Var,xGrd=None,SatA=True,SatB=True,xWin=None,xLabel=True):
    '''
    Produce two-dimensional (x,z) filled contour plot of a variable.
    Input:
        Var    : variable of interest [string]
        xGrd   : grid of horizontal (x) distances [m]
        SatA   : if True, plot orbit of satellite A
        SatB   : if True, plot orbit of satellite B
        xWin   : horizontal (x) window to select data [m]
        xLabel : if True, add xlabel
    To be provided through DIPContBas: auxiliary plot parameters.
    '''
    #.. collect imports from DIPContBas
    from DIPContBas import z2d,x2d,z2d_km,x2d_km
    from DIPContBas import zBot_km,zTop_km,xLef_km,xRig_km
    if SatA: from DIPContBas import zSatA,xSatA,zSatA_km,xSatA_km
    if SatB: from DIPContBas import zSatB,xSatB,zSatB_km,xSatB_km
    from DIPContBas import cGrdM
    if SatA or SatB: from DIPContBas import PltBox
    LM = len(cGrdM)
    #.. set parameters for supported variables
    if Var=='Tn':
        plt.title(r'Neutral temperature')
        Mod2d = VarModzx('Tn',z2d,x2d)
        cbar_label = r'$T_n~[\mathrm{K}]$'
    elif Var=='Nn':
        plt.title(r'Neutral density')
        Mod2d = np.log10(VarModzx('Nn',z2d,x2d))
        cbar_label = r'$\log_{10} ( N_n~[\mathrm{m}^{-3}] )$'
    elif Var=='Ne':
        plt.title(r'Electron density')
        Mod2d = VarModzx('Ne',z2d,x2d)
        cbar_label = r'$N_e~[\mathrm{m}^{-3}]$'
    elif Var=='Ti':
        plt.title(r'Ion temperature')
        Mod2d = VarModzx('Ti',z2d,x2d)
        cbar_label = r'$T_i~[\mathrm{K}]$'
    elif Var=='FCin':
        plt.title(r'Ion-neutral collision frequency')
        Mod2d = np.log10(VarModzx('FCin',z2d,x2d))
        cbar_label = r'$\log_{10} ( \nu_{in}~[\mathrm{Hz}] )$'
    elif Var=='Cped':
        plt.title(r'Pedersen conductivity')
        Mod2d = 1e6*VarModzx('Cped',z2d,x2d)
        cbar_label = r'$\sigma_\mathrm{P}~[\mu\mathrm{S/m}]$'
    else:
        print('VarPlt2d : non-supported value of Var')
    #.. produce filled contour plot
    plt.contourf(x2d_km,z2d_km,Mod2d,20)
    cbar = plt.colorbar()
    cbar.set_label(cbar_label)
    #.. add horizontal grid positions with selection windows
    if xGrd is not None:
        xGrd_km = xGrd/1e3
        for iGrd in range(xGrd.size):
            xC_km = xGrd_km[iGrd]
            plt.plot([xC_km,xC_km],[zBot_km,zTop_km],\
                         color=cGrdM[iGrd%LM],linewidth=2,linestyle='--')
            if xWin is not None:
                xWin_km = xWin/1e3
                xM_km = xGrd_km[iGrd]-0.5*xWin_km
                xP_km = xGrd_km[iGrd]+0.5*xWin_km
                if not SatA:
                    plt.plot([xM_km,xM_km],[zBot_km,zTop_km],\
                                 color=cGrdM[iGrd%LM],linestyle=':')
                    plt.plot([xP_km,xP_km],[zBot_km,zTop_km],\
                                 color=cGrdM[iGrd%LM],linestyle=':')
                else:
                    indWin = np.abs(xSatA-xGrd[iGrd]) <= 0.5*xWin
                    xzLL_km = [xM_km,np.min(zSatA[indWin]/1e3)]
                    xzUR_km = [xP_km,np.max(zSatA[indWin]/1e3)]
                    PltBox(xzLL_km,xzUR_km,color='white')
                    if SatB:
                        indWin = np.abs(xSatB-xGrd[iGrd]) <= 0.5*xWin
                        xzLL_km = [xM_km,np.min(zSatB[indWin]/1e3)]
                        xzUR_km = [xP_km,np.max(zSatB[indWin]/1e3)]
                        PltBox(xzLL_km,xzUR_km,color='white')
    #.. plot satellite orbits
    if SatA: plt.plot(xSatA_km,zSatA_km,color='white',\
                          linewidth=2,linestyle='--')
    if SatB: plt.plot(xSatB_km,zSatB_km,color='white',\
                          linewidth=2,linestyle='--')
    #.. add axes annotations
    if xLabel: plt.xlabel(r'North-south coordinate $x$ [km]')
    plt.xlim([xLef_km,xRig_km])
    plt.ylabel(r'Altitude $z$ [km]')
    plt.ylim([zBot_km,zTop_km])
    return 'VarPlt2d() successful.'

def PltMod1d(Var,xGrd,xWin=None):
    '''
    Produce altitude profiles of a variable at all horizontal grid 
    positions and at the boundaries of the selection windows.
    Input:
        Var    : variable of interest [string]
        xGrd   : grid of horizontal (x) distances [m]
        xWin   : horizontal (x) window to select data [m]
    To be provided through DIPContBas: auxiliary plot parameters.
    '''
    #.. collect imports from DIPContBas
    from DIPContBas import z1d,z1d_km
    from DIPContBas import zBot_km,zTop_km
    from DIPContBas import cGrdM
    LM = len(cGrdM)
    #.. draw titles and x-axis labels for supported variables
    if Var=='Tn':
        plt.title('Sample profiles of neutral temperature')
        plt.xlabel(r'$T_n~[\mathrm{K}]$')
    elif Var=='Nn':
        plt.title(r'Sample profiles of neutral density')
        plt.xlabel(r'$\log_{10} ( N_n~[\mathrm{m}^{-3}] )$')
    elif Var=='Ne':
        plt.title(r'Sample profiles of electron density')
        plt.xlabel(r'$N_e~[\mathrm{m}^{-3}]$')
    elif Var=='Ti':
        plt.title('Sample profiles of ion temperature')
        plt.xlabel(r'$T_i~[\mathrm{K}]$')
    elif Var=='FCin':
        plt.title('Sample profiles of ion-neutral collision frequency')
        plt.xlabel(r'$\log_{10} ( \nu_{in}~[\mathrm{Hz}] )$')
    elif Var=='Cped':
        plt.title('Sample profiles of Pedersen conductivity')
        plt.xlabel(r'$\sigma_\mathrm{P}~[\mu\mathrm{S/m}]$')
    else:
        print('VarPlt1d : non-supported value of Var')
    #.. plot profiles at all positions of the horizontal grid
    for iGrd in range(xGrd.size):
        ModGrd = VarModzx(Var,z1d,xGrd[iGrd])
        if Var in ['Nn','FCin']: ModGrd = np.log10(ModGrd)
        if Var=='Cped': ModGrd = 1e6*ModGrd
        plt.plot(ModGrd,z1d_km,color=cGrdM[iGrd%LM],linewidth=2)
        #.. plot profiles at selection window boundaries
        if xWin is not None:
            ModGrd = VarModzx(Var,z1d,xGrd[iGrd]-0.5*xWin)
            if Var in ['Nn','FCin']: ModGrd = np.log10(ModGrd)
            if Var=='Cped': ModGrd = 1e6*ModGrd
            plt.plot(ModGrd,z1d_km,color=cGrdM[iGrd%LM],linestyle=':')
            ModGrd = VarModzx(Var,z1d,xGrd[iGrd]+0.5*xWin)
            if Var in ['Nn','FCin']: ModGrd = np.log10(ModGrd)
            if Var=='Cped': ModGrd = 1e6*ModGrd
            plt.plot(ModGrd,z1d_km,color=cGrdM[iGrd%LM],linestyle=':')
    #..  add y-axis labels
    plt.ylabel(r'Altitude $z$ [km]')
    plt.ylim([zBot_km,zTop_km])
    plt.grid()
    return 'VarPlt1d() successful.'

#################################
### End of file DIPContMod.py ###
#################################

